<?php

class AuthController
{
    private $pdo;
    private $user;
    private $organization;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->user = new User($pdo);
        $this->organization = new Organization($pdo);
        Session::start();
    }

    // Show login form
    public function showLogin()
    {
        if (Session::isAuthenticated()) {
            header('Location: index.php');
            exit;
        }

        $title = "Login";
        include __DIR__ . '/../../views/layouts/auth.php';
        include __DIR__ . '/../../views/auth/login.php';
        include __DIR__ . '/../../views/layouts/auth-footer.php';
    }

    // Handle login
    public function login()
    {
        // Verify CSRF token
        Security::verifyCSRF();
        
        // Rate limiting for login attempts
        require_once __DIR__ . '/../Helpers/Env.php';
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $rateLimitKey = 'login_' . $ip;
        
        $loginAttempts = (int)Env::get('RATE_LIMIT_LOGIN_ATTEMPTS', '5');
        $loginWindow = (int)Env::get('RATE_LIMIT_LOGIN_WINDOW', '900');
        $windowMinutes = round($loginWindow / 60);
        
        if (!Security::checkRateLimit($rateLimitKey, $loginAttempts, $loginWindow)) {
            Session::flash('error', "Too many login attempts. Please try again in {$windowMinutes} minutes.");
            Functions::logError("Login rate limit exceeded for IP: $ip");
            header('Location: login.php');
            exit;
        }
        
        $email = Security::sanitizeString($_POST['email'] ?? '', 255);
        $password = $_POST['password'] ?? '';
        $remember = isset($_POST['remember']);

        if (empty($email) || empty($password)) {
            Session::flash('error', 'Email and password are required.');
            header('Location: login.php');
            exit;
        }

        // Validate email format
        if (!Security::validateEmail($email)) {
            Session::flash('error', 'Invalid email format.');
            header('Location: login.php');
            exit;
        }

        $user = $this->user->findByEmail($email);

        if (!$user || !$this->user->verifyPassword($password, $user['password_hash'])) {
            Session::flash('error', 'Invalid email or password.');
            Functions::logError("Failed login attempt for email: $email from IP: $ip");
            header('Location: login.php');
            exit;
        }

        // Clear rate limit on successful login
        Security::clearRateLimit($rateLimitKey);
        
        // Regenerate session ID to prevent session fixation
        Security::regenerateSession();
        
        // Set user session
        Session::setUser($user);
        
        // Set additional security session data
        Session::set('login_time', time());
        Session::set('login_ip', $ip);
        Session::set('user_agent', $_SERVER['HTTP_USER_AGENT'] ?? '');

        // Get user's organizations
        $organizations = $this->user->getOrganizations($user['id']);

        if (!empty($organizations)) {
            // Set first organization as active
            Session::setOrganization($organizations[0]);
        } else {
            // No organizations found - user needs to create one
            // Don't redirect to settings here to avoid loop, let index.php handle it
        }

        // Remember me
        if ($remember) {
            $token = bin2hex(random_bytes(32));
            $this->user->updateRememberToken($user['id'], $token);
            $cookieParams = [
                'expires' => time() + (86400 * 30), // 30 days
                'path' => '/',
                'httponly' => true,
                'secure' => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'),
                'samesite' => 'Strict'
            ];
            setcookie('remember_token', $token, $cookieParams);
        }

        Functions::logError("Successful login for user: {$user['email']} from IP: $ip");
        Session::flash('success', 'Welcome back, ' . htmlspecialchars($user['name'], ENT_QUOTES, 'UTF-8') . '!');
        
        // Redirect to settings if no organization, otherwise to dashboard
        if (empty($organizations)) {
            header('Location: settings.php');
        } else {
            header('Location: index.php');
        }
        exit;
    }

    // Show register form
    public function showRegister()
    {
        if (Session::isAuthenticated()) {
            header('Location: index.php');
            exit;
        }

        $title = "Register";
        include __DIR__ . '/../../views/layouts/auth.php';
        include __DIR__ . '/../../views/auth/register.php';
        include __DIR__ . '/../../views/layouts/auth-footer.php';
    }

    // Handle registration
    public function register()
    {
        // Verify CSRF token
        Security::verifyCSRF();
        
        // Rate limiting for registration attempts
        require_once __DIR__ . '/../Helpers/Env.php';
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $rateLimitKey = 'register_' . $ip;
        
        $registerAttempts = (int)Env::get('RATE_LIMIT_REGISTER_ATTEMPTS', '3');
        $registerWindow = (int)Env::get('RATE_LIMIT_REGISTER_WINDOW', '3600');
        $windowHours = round($registerWindow / 3600);
        
        if (!Security::checkRateLimit($rateLimitKey, $registerAttempts, $registerWindow)) {
            Session::flash('error', "Too many registration attempts. Please try again in {$windowHours} hour(s).");
            Functions::logError("Registration rate limit exceeded for IP: $ip");
            header('Location: register.php');
            exit;
        }
        
        $name = Security::sanitizeString($_POST['name'] ?? '', 255);
        $email = Security::sanitizeString($_POST['email'] ?? '', 255);
        $password = $_POST['password'] ?? '';
        $password_confirm = $_POST['password_confirm'] ?? '';
        $organization_name = Security::sanitizeString($_POST['organization_name'] ?? '', 255);

        // Validation
        $errors = [];

        if (empty($name)) {
            $errors[] = 'Name is required.';
        }

        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Valid email is required.';
        } else {
            $existing = $this->user->findByEmail($email);
            if ($existing) {
                $errors[] = 'Email already registered.';
            }
        }

        if (empty($password) || strlen($password) < 8) {
            $errors[] = 'Password must be at least 8 characters.';
        }

        if ($password !== $password_confirm) {
            $errors[] = 'Passwords do not match.';
        }

        if (empty($organization_name)) {
            $errors[] = 'Organization name is required.';
        }

        if (!empty($errors)) {
            Session::flash('errors', $errors);
            header('Location: register.php');
            exit;
        }

        // Create user
        $userId = $this->user->create([
            'name' => $name,
            'email' => $email,
            'password' => $password
        ]);

        // Create organization
        $slug = Organization::generateSlug($organization_name);
        // Ensure unique slug
        $originalSlug = $slug;
        $counter = 1;
        while ($this->organization->findBySlug($slug)) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        $organizationId = $this->organization->create([
            'name' => $organization_name,
            'slug' => $slug
        ]);

        // Add user as owner
        $this->organization->addUser($organizationId, $userId, 'owner');

        // Set sessions
        $user = $this->user->findById($userId);
        Session::setUser($user);

        $organization = $this->organization->findById($organizationId);
        Session::setOrganization($organization);

        Session::flash('success', 'Account created successfully! Welcome to Payment System.');
        header('Location: index.php');
        exit;
    }

    // Handle logout
    public function logout()
    {
        Session::destroy();
        setcookie('remember_token', '', time() - 3600, '/');
        Session::flash('success', 'You have been logged out.');
        header('Location: login.php');
        exit;
    }
}

