<?php

class AdminPlansController
{
    private $pdo;
    private $plan;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->plan = new Plan($pdo);
    }

    // Show all plans
    public function index()
    {
        $plans = $this->plan->all(false); // Get all plans including inactive
        
        $title = "Manage Plans";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/admin/plans/index.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Show create form
    public function create()
    {
        $title = "Create Plan";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/admin/plans/form.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Show edit form
    public function edit()
    {
        $id = Security::sanitizeString($_GET['id'] ?? '');
        if (!$id) {
            Session::flash('error', 'Plan ID is required.');
            header('Location: admin-plans.php');
            exit;
        }

        $plan = $this->plan->findById($id);
        if (!$plan) {
            Session::flash('error', 'Plan not found.');
            header('Location: admin-plans.php');
            exit;
        }

        $title = "Edit Plan";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/admin/plans/form.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Store new plan
    public function store()
    {
        Security::verifyCSRF();
        $data = $this->validateAndPrepareData($_POST);
        
        if (isset($data['error'])) {
            Session::flash('error', $data['error']);
            header('Location: admin-plans.php?action=create');
            exit;
        }

        // Generate slug from name
        $slug = $this->generateSlug($data['name']);
        
        // Ensure unique slug
        $originalSlug = $slug;
        $counter = 1;
        while ($this->plan->findBySlug($slug)) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        $data['slug'] = $slug;

        if ($this->plan->create($data)) {
            Session::flash('success', 'Plan created successfully!');
            header('Location: admin-plans.php');
        } else {
            Session::flash('error', 'Failed to create plan. Please try again.');
            header('Location: admin-plans.php?action=create');
        }
        exit;
    }

    // Update existing plan
    public function update()
    {
        Security::verifyCSRF();
        $id = Security::sanitizeString($_POST['id'] ?? '');
        if (!$id) {
            Session::flash('error', 'Plan ID is required.');
            header('Location: admin-plans.php');
            exit;
        }

        $plan = $this->plan->findById($id);
        if (!$plan) {
            Session::flash('error', 'Plan not found.');
            header('Location: admin-plans.php');
            exit;
        }

        $data = $this->validateAndPrepareData($_POST);
        
        if (isset($data['error'])) {
            Session::flash('error', $data['error']);
            header('Location: admin-plans.php?action=edit&id=' . $id);
            exit;
        }

        // Update slug if name changed
        if (isset($data['name']) && $data['name'] !== $plan['name']) {
            $slug = $this->generateSlug($data['name']);
            $originalSlug = $slug;
            $counter = 1;
            while ($existing = $this->plan->findBySlug($slug)) {
                if ($existing['id'] != (int)$id) {
                    $slug = $originalSlug . '-' . $counter;
                    $counter++;
                } else {
                    break;
                }
            }
            $data['slug'] = $slug;
        }

        if ($this->plan->update($id, $data)) {
            Session::flash('success', 'Plan updated successfully!');
            header('Location: admin-plans.php');
        } else {
            Session::flash('error', 'Failed to update plan. Please try again.');
            header('Location: admin-plans.php?action=edit&id=' . $id);
        }
        exit;
    }

    // Delete plan
    public function delete()
    {
        Security::verifyCSRF();
        $id = Security::sanitizeString($_POST['id'] ?? '');
        if (!$id) {
            Session::flash('error', 'Plan ID is required.');
            header('Location: admin-plans.php');
            exit;
        }

        if ($this->plan->delete($id)) {
            Session::flash('success', 'Plan deleted successfully!');
        } else {
            Session::flash('error', 'Failed to delete plan. It may be in use by active subscriptions.');
        }

        header('Location: admin-plans.php');
        exit;
    }

    // Toggle plan active status
    public function toggle()
    {
        Security::verifyCSRF();
        $id = Security::sanitizeString($_POST['id'] ?? '');
        if (!$id) {
            Session::flash('error', 'Plan ID is required.');
            header('Location: admin-plans.php');
            exit;
        }

        $plan = $this->plan->findById($id);
        if (!$plan) {
            Session::flash('error', 'Plan not found.');
            header('Location: admin-plans.php');
            exit;
        }

        $newStatus = $plan['is_active'] ? 0 : 1;
        if ($this->plan->update($id, ['is_active' => $newStatus])) {
            Session::flash('success', 'Plan status updated successfully!');
        } else {
            Session::flash('error', 'Failed to update plan status.');
        }

        header('Location: admin-plans.php');
        exit;
    }

    // Validate and prepare form data
    private function validateAndPrepareData($post)
    {
        $name = trim($post['name'] ?? '');
        $description = trim($post['description'] ?? '');
        $priceMonthly = $post['price_monthly'] ?? 0;
        $priceAnnual = $post['price_annual'] ?? 0;
        $features = $post['features'] ?? [];
        $maxTransactions = !empty($post['max_transactions']) ? (int)$post['max_transactions'] : null;
        $maxUsers = !empty($post['max_users']) ? (int)$post['max_users'] : null;
        $maxOrganizations = !empty($post['max_organizations']) ? (int)$post['max_organizations'] : 1;
        $isActive = isset($post['is_active']) ? (int)$post['is_active'] : 1;
        $sortOrder = !empty($post['sort_order']) ? (int)$post['sort_order'] : 0;

        if (empty($name)) {
            return ['error' => 'Plan name is required.'];
        }

        if (!is_numeric($priceMonthly) || $priceMonthly < 0) {
            return ['error' => 'Monthly price must be a valid number.'];
        }

        if (!is_numeric($priceAnnual) || $priceAnnual < 0) {
            return ['error' => 'Annual price must be a valid number.'];
        }

        // Process features - handle both array and newline-separated string
        if (is_string($features)) {
            $features = array_filter(array_map('trim', explode("\n", $features)));
        } elseif (!is_array($features)) {
            $features = [];
        }

        return [
            'name' => $name,
            'description' => $description ?: null,
            'price_monthly' => (float)$priceMonthly,
            'price_annual' => (float)$priceAnnual,
            'features' => $features,
            'max_transactions' => $maxTransactions,
            'max_users' => $maxUsers,
            'max_organizations' => $maxOrganizations,
            'is_active' => $isActive,
            'sort_order' => $sortOrder
        ];
    }

    // Generate URL-friendly slug
    private function generateSlug($name)
    {
        $slug = strtolower(trim($name));
        $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
        $slug = preg_replace('/-+/', '-', $slug);
        $slug = trim($slug, '-');
        return $slug;
    }
}

