<?php

class Security
{
    /**
     * Generate CSRF token
     */
    public static function generateCSRFToken()
    {
        Session::start();
        if (!Session::has('csrf_token')) {
            Session::set('csrf_token', bin2hex(random_bytes(32)));
        }
        return Session::get('csrf_token');
    }

    /**
     * Verify CSRF token
     */
    public static function verifyCSRFToken($token)
    {
        Session::start();
        $sessionToken = Session::get('csrf_token');
        return $sessionToken && hash_equals($sessionToken, $token);
    }

    /**
     * Generate CSRF token field for forms
     */
    public static function csrfField()
    {
        try {
            $token = self::generateCSRFToken();
            return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
        } catch (Exception $e) {
            // Fallback: return empty string if there's an error
            return '';
        }
    }

    /**
     * Verify CSRF token from POST request
     */
    public static function verifyCSRF()
    {
        $token = $_POST['csrf_token'] ?? '';
        if (!self::verifyCSRFToken($token)) {
            http_response_code(403);
            die('CSRF token validation failed. Please refresh the page and try again.');
        }
    }

    /**
     * Sanitize output to prevent XSS
     */
    public static function escape($value, $flags = ENT_QUOTES, $encoding = 'UTF-8')
    {
        if (is_null($value)) {
            return '';
        }
        return htmlspecialchars((string)$value, $flags, $encoding);
    }

    /**
     * Validate and sanitize email
     */
    public static function validateEmail($email)
    {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Validate integer
     */
    public static function validateInt($value, $min = null, $max = null)
    {
        if (!is_numeric($value)) {
            return false;
        }
        $int = (int)$value;
        if ($min !== null && $int < $min) {
            return false;
        }
        if ($max !== null && $int > $max) {
            return false;
        }
        return true;
    }

    /**
     * Rate limiting - file-based implementation for persistence across requests
     */
    public static function checkRateLimit($key, $maxAttempts = null, $window = null)
    {
        try {
            require_once __DIR__ . '/Env.php';
            
            // Use defaults from .env if not provided
            if ($maxAttempts === null) {
                $maxAttempts = (int)Env::get('RATE_LIMIT_DEFAULT_ATTEMPTS', '5');
            }
            if ($window === null) {
                $window = (int)Env::get('RATE_LIMIT_DEFAULT_WINDOW', '300');
            }
            
            $cacheDir = __DIR__ . '/../../storage/cache';
            if (!is_dir($cacheDir)) {
                @mkdir($cacheDir, 0755, true);
            }
            
            $rateLimitFile = $cacheDir . '/rate_limit_' . md5($key) . '.json';
            $now = time();
            
            // Read existing data
            $data = [
                'attempts' => 0,
                'window_start' => $now,
                'locked_until' => 0
            ];
            
            if (file_exists($rateLimitFile)) {
                $content = @file_get_contents($rateLimitFile);
                if ($content !== false) {
                    $stored = json_decode($content, true);
                    if ($stored && is_array($stored)) {
                        $data = $stored;
                    }
                }
            }
            
            // Check if still locked
            if (isset($data['locked_until']) && $data['locked_until'] > $now) {
                $remaining = $data['locked_until'] - $now;
                if (class_exists('Functions')) {
                    Functions::logError("Rate limit exceeded for key: $key. Locked for $remaining seconds.");
                }
                return false;
            }
            
            // Reset if window expired
            if (isset($data['window_start']) && ($now - $data['window_start'] > $window)) {
                $data['attempts'] = 0;
                $data['window_start'] = $now;
                $data['locked_until'] = 0;
            }
            
            // Check if limit exceeded
            if (isset($data['attempts']) && $data['attempts'] >= $maxAttempts) {
                // Lock for the remaining window time
                $data['locked_until'] = ($data['window_start'] ?? $now) + $window;
                @file_put_contents($rateLimitFile, json_encode($data), LOCK_EX);
                if (class_exists('Functions')) {
                    Functions::logError("Rate limit exceeded for key: $key. Locked until " . date('Y-m-d H:i:s', $data['locked_until']));
                }
                return false;
            }
            
            // Increment attempts
            $data['attempts'] = ($data['attempts'] ?? 0) + 1;
            $data['window_start'] = $data['window_start'] ?? $now;
            @file_put_contents($rateLimitFile, json_encode($data), LOCK_EX);
            
            return true;
        } catch (Exception $e) {
            // On error, allow the request (fail open for availability)
            return true;
        }
    }
    
    /**
     * Clear rate limit for a key (useful after successful login)
     */
    public static function clearRateLimit($key)
    {
        try {
            $rateLimitFile = __DIR__ . '/../../storage/cache/rate_limit_' . md5($key) . '.json';
            if (file_exists($rateLimitFile)) {
                @unlink($rateLimitFile);
            }
        } catch (Exception $e) {
            // Silently fail
        }
    }

    /**
     * Set security headers
     */
    public static function setSecurityHeaders()
    {
        // Prevent clickjacking
        header('X-Frame-Options: SAMEORIGIN');
        
        // Prevent MIME type sniffing
        header('X-Content-Type-Options: nosniff');
        
        // Enable XSS protection
        header('X-XSS-Protection: 1; mode=block');
        
        // Referrer policy
        header('Referrer-Policy: strict-origin-when-cross-origin');
        
        // Content Security Policy
        require_once __DIR__ . '/Env.php';
        $cspPolicy = Env::get('CSP_POLICY', '');
        
        if (empty($cspPolicy)) {
            // Default CSP: Allow Tailwind CDN and other necessary external resources
            $cspPolicy = "default-src 'self'; script-src 'self' 'unsafe-inline' https://cdn.tailwindcss.com https://unpkg.com; style-src 'self' 'unsafe-inline' https://cdn.tailwindcss.com; img-src 'self' data: https:; font-src 'self' data:;";
        }
        
        header("Content-Security-Policy: {$cspPolicy}");
    }

    /**
     * Regenerate session ID to prevent session fixation
     */
    public static function regenerateSession()
    {
        Session::start();
        session_regenerate_id(true);
    }
    
    /**
     * Sanitize input to prevent XSS and injection
     */
    public static function sanitizeInput($value)
    {
        if (is_array($value)) {
            return array_map([self::class, 'sanitizeInput'], $value);
        }
        return htmlspecialchars(trim((string)$value), ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * Validate and sanitize string input
     */
    public static function sanitizeString($value, $maxLength = null)
    {
        $value = trim((string)$value);
        if ($maxLength !== null && strlen($value) > $maxLength) {
            $value = substr($value, 0, $maxLength);
        }
        return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * Validate file path to prevent directory traversal
     */
    public static function validateFilePath($path, $baseDir)
    {
        $realBase = realpath($baseDir);
        $realPath = realpath($baseDir . '/' . $path);
        
        if ($realPath === false || strpos($realPath, $realBase) !== 0) {
            return false;
        }
        
        return $realPath;
    }
}
