<?php

class Organization
{
    private $pdo;
    private $table = "organizations";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Create new organization
    public function create($data)
    {
        $sql = "INSERT INTO {$this->table} (name, slug, telegram_bot_token, settings) 
                VALUES (:name, :slug, :telegram_bot_token, :settings)";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':name' => $data['name'],
            ':slug' => $data['slug'],
            ':telegram_bot_token' => $data['telegram_bot_token'] ?? null,
            ':settings' => isset($data['settings']) ? json_encode($data['settings']) : null
        ]);
        
        return $this->pdo->lastInsertId();
    }

    // Find by ID
    public function findById($id)
    {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch();
    }

    // Find by slug
    public function findBySlug($slug)
    {
        $sql = "SELECT * FROM {$this->table} WHERE slug = :slug LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':slug' => $slug]);
        return $stmt->fetch();
    }

    // Update organization
    public function update($id, $data)
    {
        $fields = [];
        $params = [':id' => $id];

        if (isset($data['name'])) {
            $fields[] = "name = :name";
            $params[':name'] = $data['name'];
        }
        if (isset($data['slug'])) {
            $fields[] = "slug = :slug";
            $params[':slug'] = $data['slug'];
        }
        if (isset($data['telegram_bot_token'])) {
            $fields[] = "telegram_bot_token = :telegram_bot_token";
            $params[':telegram_bot_token'] = $data['telegram_bot_token'];
        }
        if (isset($data['settings'])) {
            $fields[] = "settings = :settings";
            $params[':settings'] = json_encode($data['settings']);
        }

        if (empty($fields)) {
            return false;
        }

        $sql = "UPDATE {$this->table} SET " . implode(", ", $fields) . " WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    // Delete organization
    public function delete($id)
    {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

    // Get all organizations
    public function all($limit = null, $offset = 0)
    {
        $sql = "SELECT * FROM {$this->table} ORDER BY created_at DESC";
        if ($limit !== null) {
            $sql .= " LIMIT :limit OFFSET :offset";
        }
        
        $stmt = $this->pdo->prepare($sql);
        if ($limit !== null) {
            $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        }
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // Get organization users
    public function getUsers($organizationId)
    {
        $sql = "SELECT u.*, ou.role, ou.created_at as joined_at
                FROM users u
                INNER JOIN organization_users ou ON u.id = ou.user_id
                WHERE ou.organization_id = :organization_id
                ORDER BY ou.role, ou.created_at DESC";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':organization_id' => $organizationId]);
        return $stmt->fetchAll();
    }

    // Add user to organization
    public function addUser($organizationId, $userId, $role = 'member')
    {
        $sql = "INSERT INTO organization_users (organization_id, user_id, role) 
                VALUES (:organization_id, :user_id, :role)
                ON DUPLICATE KEY UPDATE role = :role";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':organization_id' => $organizationId,
            ':user_id' => $userId,
            ':role' => $role
        ]);
    }

    // Remove user from organization
    public function removeUser($organizationId, $userId)
    {
        $sql = "DELETE FROM organization_users 
                WHERE organization_id = :organization_id AND user_id = :user_id";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':organization_id' => $organizationId,
            ':user_id' => $userId
        ]);
    }

    // Update user role
    public function updateUserRole($organizationId, $userId, $role)
    {
        $sql = "UPDATE organization_users 
                SET role = :role 
                WHERE organization_id = :organization_id AND user_id = :user_id";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':role' => $role,
            ':organization_id' => $organizationId,
            ':user_id' => $userId
        ]);
    }

    // Generate slug from name
    public static function generateSlug($name)
    {
        $slug = strtolower(trim($name));
        $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
        $slug = preg_replace('/-+/', '-', $slug);
        $slug = trim($slug, '-');
        return $slug;
    }
}

