<?php

// Error handling configuration
require_once __DIR__ . '/../app/Config/app.php';
$app = require __DIR__ . '/../app/Config/app.php';

// Set timezone from configuration
if (isset($app['timezone']) && !empty($app['timezone'])) {
    date_default_timezone_set($app['timezone']);
}

// Set error reporting based on environment
if ($app['app_env'] === 'production' || !$app['app_debug']) {
    // Production: Hide errors from users
    ini_set('display_errors', '0');
    ini_set('display_startup_errors', '0');
    error_reporting(E_ALL);
    ini_set('log_errors', '1');
    ini_set('error_log', __DIR__ . '/../storage/logs/system.log');
} else {
    // Development: Show errors
    ini_set('display_errors', '1');
    ini_set('display_startup_errors', '1');
    error_reporting(E_ALL);
    ini_set('log_errors', '1');
    ini_set('error_log', __DIR__ . '/../storage/logs/system.log');
}

// Set custom error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) use ($app) {
    $logFile = __DIR__ . '/../storage/logs/system.log';
    $message = sprintf(
        "[%s] PHP Error [%d]: %s in %s on line %d",
        date('Y-m-d H:i:s'),
        $errno,
        $errstr,
        $errfile,
        $errline
    );
    file_put_contents($logFile, $message . "\n", FILE_APPEND);
    
    // In production, don't display errors
    if ($app['app_env'] === 'production' || !$app['app_debug']) {
        return true; // Suppress error display
    }
    return false; // Let PHP handle it normally
});

// Set exception handler
set_exception_handler(function($exception) use ($app) {
    $logFile = __DIR__ . '/../storage/logs/system.log';
    $message = sprintf(
        "[%s] Uncaught Exception: %s in %s on line %d\nStack trace:\n%s",
        date('Y-m-d H:i:s'),
        $exception->getMessage(),
        $exception->getFile(),
        $exception->getLine(),
        $exception->getTraceAsString()
    );
    file_put_contents($logFile, $message . "\n", FILE_APPEND);
    
    // In production, show generic error
    if ($app['app_env'] === 'production' || !$app['app_debug']) {
        http_response_code(500);
        die('An error occurred. Please contact the administrator.');
    } else {
        // Development: show full error
        die($message);
    }
});

// Autoload all necessary classes
spl_autoload_register(function ($class) {
    $base_path = __DIR__ . '/../app/';

    $paths = [
        "Controllers/$class.php",
        "Models/$class.php",
        "Helpers/$class.php",
        "Services/$class.php",
        "Auth/$class.php",
        "Auth/Controllers/$class.php",
        "Middleware/$class.php",
    ];

    foreach ($paths as $path) {
        $full = $base_path . $path;
        if (file_exists($full)) {
            require_once $full;
            return;
        }
    }
});

// Load essential helpers first
require_once __DIR__ . '/../app/Helpers/Session.php';
require_once __DIR__ . '/../app/Helpers/Security.php';

// Set security headers
Security::setSecurityHeaders();

// Load shared configurations and database connection
$pdo = require __DIR__ . '/../app/Config/db.php';
