# Quick Implementation Guide
## Security & Performance Fixes

This guide helps you quickly implement the remaining security and performance improvements.

---

## 🚀 Quick Start - Critical Fixes

### 1. Add Database Indexes (5 minutes)

Run the migration to add performance indexes:

```bash
# Option 1: Using MySQL command line
mysql -u root -p sumtel < database/migrations/add_performance_indexes.sql

# Option 2: Using phpMyAdmin
# 1. Open phpMyAdmin
# 2. Select your database (sumtel)
# 3. Click "SQL" tab
# 4. Paste contents of database/migrations/add_performance_indexes.sql
# 5. Click "Go"
```

**Expected Result:** Faster queries, especially for transactions and webhook lookups.

---

### 2. Add CSRF Protection to Forms (15-20 minutes)

#### Step 1: Update login form

**File:** `views/auth/login.php`

Add this inside the `<form>` tag:
```php
<?= Security::csrfField() ?>
```

Add this at the top of `app/Auth/AuthController.php` in the `login()` method:
```php
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    Security::verifyCSRF();
    // ... rest of login code
}
```

#### Step 2: Update register form

**File:** `views/auth/register.php`

Add CSRF field in form:
```php
<?= Security::csrfField() ?>
```

Add verification in `app/Auth/AuthController.php` in the `register()` method:
```php
Security::verifyCSRF();
```

#### Step 3: Update all other forms

Repeat for:
- `views/settings/index.php` (3 forms)
- `views/admin/users/form.php`
- `views/admin/plans/form.php`
- Password reset forms

**Pattern:**
1. Add `<?= Security::csrfField() ?>` in form
2. Add `Security::verifyCSRF();` at start of POST handler

---

### 3. Add Rate Limiting (10 minutes)

#### For Login

**File:** `app/Auth/AuthController.php`

Update `login()` method:
```php
public function login()
{
    // Rate limiting - 5 attempts per 15 minutes per IP
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $rateLimitKey = 'login_' . $ip;
    
    if (!Security::checkRateLimit($rateLimitKey, 5, 900)) {
        Session::flash('error', 'Too many login attempts. Please try again in 15 minutes.');
        header('Location: login.php');
        exit;
    }
    
    // ... rest of login code
    
    // On successful login, reset rate limit
    Security::checkRateLimit($rateLimitKey, 5, 900); // Reset on success
}
```

#### For Registration

**File:** `app/Auth/AuthController.php`

Update `register()` method:
```php
public function register()
{
    // Rate limiting - 3 attempts per hour per IP
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $rateLimitKey = 'register_' . $ip;
    
    if (!Security::checkRateLimit($rateLimitKey, 3, 3600)) {
        Session::flash('error', 'Too many registration attempts. Please try again later.');
        header('Location: register.php');
        exit;
    }
    
    // ... rest of registration code
}
```

---

### 4. Add Pagination to Transaction List (15 minutes)

**File:** `public/transaction-list.php`

Replace the current code with:

```php
<?php
require_once __DIR__ . '/../routes/web.php';

Authenticate::handle();

$organization = Session::getOrganization();
if (!$organization) {
    Session::flash('error', 'No organization found.');
    header('Location: index.php');
    exit;
}

$transaction = new Transaction($pdo);

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$page = max(1, $page); // Ensure page is at least 1
$perPage = 50;
$offset = ($page - 1) * $perPage;

// Get total count for pagination
$totalCount = $transaction->count(['organization_id' => $organization['id']]);
$totalPages = ceil($totalCount / $perPage);

// Get paginated data
$data = $transaction->all($perPage, $offset, $organization['id']);

$pagination = [
    'page' => $page,
    'perPage' => $perPage,
    'totalCount' => $totalCount,
    'totalPages' => $totalPages
];

include __DIR__ . '/../views/transactions/list.php';
```

**File:** `views/transactions/list.php`

Add pagination controls at the bottom:

```php
<?php if ($pagination['totalPages'] > 1): ?>
    <div class="mt-6 flex justify-between items-center">
        <div class="text-sm text-mono-600">
            Showing <?= ($pagination['page'] - 1) * $pagination['perPage'] + 1 ?> 
            to <?= min($pagination['page'] * $pagination['perPage'], $pagination['totalCount']) ?> 
            of <?= $pagination['totalCount'] ?> transactions
        </div>
        <div class="flex gap-2">
            <?php if ($pagination['page'] > 1): ?>
                <a href="?page=<?= $pagination['page'] - 1 ?>" 
                   class="px-4 py-2 bg-mono-200 text-mono-700 rounded hover:bg-mono-300">
                    Previous
                </a>
            <?php endif; ?>
            
            <?php for ($i = 1; $i <= $pagination['totalPages']; $i++): ?>
                <?php if ($i == $pagination['page']): ?>
                    <span class="px-4 py-2 bg-mono-800 text-white rounded"><?= $i ?></span>
                <?php else: ?>
                    <a href="?page=<?= $i ?>" 
                       class="px-4 py-2 bg-mono-200 text-mono-700 rounded hover:bg-mono-300">
                        <?= $i ?>
                    </a>
                <?php endif; ?>
            <?php endfor; ?>
            
            <?php if ($pagination['page'] < $pagination['totalPages']): ?>
                <a href="?page=<?= $pagination['page'] + 1 ?>" 
                   class="px-4 py-2 bg-mono-200 text-mono-700 rounded hover:bg-mono-300">
                    Next
                </a>
            <?php endif; ?>
        </div>
    </div>
<?php endif; ?>
```

---

### 5. Improve Session Security (5 minutes)

**File:** `app/Helpers/Session.php`

Add session configuration in `start()` method:

```php
public static function start()
{
    if (session_status() === PHP_SESSION_NONE) {
        // Security settings
        ini_set('session.cookie_httponly', 1);
        ini_set('session.use_strict_mode', 1);
        
        // If using HTTPS (production)
        // ini_set('session.cookie_secure', 1);
        
        // Session timeout (30 minutes)
        ini_set('session.gc_maxlifetime', 1800);
        
        session_start();
    }
}
```

**File:** `app/Auth/AuthController.php`

Add session regeneration on login:

```php
// After successful login, before setting user
Security::regenerateSession();
Session::setUser($user);
```

---

### 6. Add Input Validation Helper (10 minutes)

**File:** `app/Helpers/Validator.php` (NEW FILE)

```php
<?php

class Validator
{
    public static function validate($data, $rules)
    {
        $errors = [];
        
        foreach ($rules as $field => $ruleSet) {
            $value = $data[$field] ?? null;
            $rulesArray = explode('|', $ruleSet);
            
            foreach ($rulesArray as $rule) {
                if ($rule === 'required' && empty($value)) {
                    $errors[$field] = ucfirst($field) . ' is required.';
                    break;
                }
                
                if ($rule === 'email' && !empty($value) && !Security::validateEmail($value)) {
                    $errors[$field] = ucfirst($field) . ' must be a valid email.';
                    break;
                }
                
                if (strpos($rule, 'min:') === 0) {
                    $min = (int)substr($rule, 4);
                    if (!empty($value) && strlen($value) < $min) {
                        $errors[$field] = ucfirst($field) . " must be at least {$min} characters.";
                        break;
                    }
                }
                
                if (strpos($rule, 'max:') === 0) {
                    $max = (int)substr($rule, 4);
                    if (!empty($value) && strlen($value) > $max) {
                        $errors[$field] = ucfirst($field) . " must not exceed {$max} characters.";
                        break;
                    }
                }
            }
        }
        
        return $errors;
    }
}
```

**Usage Example:**

```php
$errors = Validator::validate($_POST, [
    'name' => 'required|min:3|max:100',
    'email' => 'required|email',
    'password' => 'required|min:8'
]);

if (!empty($errors)) {
    Session::flash('errors', $errors);
    header('Location: register.php');
    exit;
}
```

---

### 7. Add Simple Caching (15 minutes)

**File:** `app/Helpers/Cache.php` (NEW FILE)

```php
<?php

class Cache
{
    private static $cacheDir = __DIR__ . '/../../storage/cache/';
    
    public static function get($key, $default = null)
    {
        $file = self::getCacheFile($key);
        
        if (!file_exists($file)) {
            return $default;
        }
        
        $data = unserialize(file_get_contents($file));
        
        // Check expiration
        if ($data['expires'] < time()) {
            unlink($file);
            return $default;
        }
        
        return $data['value'];
    }
    
    public static function set($key, $value, $ttl = 300)
    {
        if (!is_dir(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }
        
        $file = self::getCacheFile($key);
        $data = [
            'value' => $value,
            'expires' => time() + $ttl
        ];
        
        file_put_contents($file, serialize($data));
    }
    
    public static function forget($key)
    {
        $file = self::getCacheFile($key);
        if (file_exists($file)) {
            unlink($file);
        }
    }
    
    public static function flush()
    {
        $files = glob(self::$cacheDir . '*.cache');
        foreach ($files as $file) {
            unlink($file);
        }
    }
    
    private static function getCacheFile($key)
    {
        return self::$cacheDir . md5($key) . '.cache';
    }
}
```

**Usage in Dashboard:**

```php
// Cache dashboard stats for 5 minutes
$cacheKey = 'dashboard_stats_' . $organization['id'];
$dashboardStats = Cache::get($cacheKey);

if (!$dashboardStats) {
    $dashboardStats = $summary->getDashboardStats($organization['id']);
    Cache::set($cacheKey, $dashboardStats, 300);
}
```

---

## 📋 Implementation Checklist

### Critical (Do First)
- [ ] Run database indexes migration
- [ ] Add CSRF protection to login form
- [ ] Add CSRF protection to register form
- [ ] Add rate limiting to login
- [ ] Add pagination to transaction list

### High Priority
- [ ] Add CSRF protection to all other forms
- [ ] Improve session security
- [ ] Add input validation
- [ ] Add caching for dashboard

### Medium Priority
- [ ] Add authorization checks
- [ ] Improve error handling
- [ ] Add monitoring/logging

---

## 🔍 Testing Your Changes

### Test CSRF Protection
1. Login to the application
2. Open browser developer tools (F12)
3. Go to Network tab
4. Try to submit a form
5. Copy the CSRF token from the request
6. Try to submit the form again with an old token
7. Should receive 403 error

### Test Rate Limiting
1. Try logging in 6 times with wrong password
2. 6th attempt should be blocked
3. Wait 15 minutes, should work again

### Test Performance
1. Check database query count (use MySQL query log)
2. Verify indexes are being used (EXPLAIN queries)
3. Test webhook with many organizations
4. Should see 1 query instead of 100+

---

## 📞 Need Help?

Refer to:
- `SECURITY_PERFORMANCE_REPORT.md` - Full detailed report
- `app/Helpers/Security.php` - Security utility functions
- Database migration files in `database/migrations/`

---

**Good luck with your implementation!** 🚀
