<?php

class AuthController
{
    private $pdo;
    private $user;
    private $organization;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->user = new User($pdo);
        $this->organization = new Organization($pdo);
        Session::start();
    }

    // Show login form
    public function showLogin()
    {
        if (Session::isAuthenticated()) {
            header('Location: index.php');
            exit;
        }

        $title = "Login";
        include __DIR__ . '/../../views/layouts/auth.php';
        include __DIR__ . '/../../views/auth/login.php';
        include __DIR__ . '/../../views/layouts/auth-footer.php';
    }

    // Handle login
    public function login()
    {
        $email = $_POST['email'] ?? '';
        $password = $_POST['password'] ?? '';
        $remember = isset($_POST['remember']);

        if (empty($email) || empty($password)) {
            Session::flash('error', 'Email and password are required.');
            header('Location: login.php');
            exit;
        }

        $user = $this->user->findByEmail($email);

        if (!$user || !$this->user->verifyPassword($password, $user['password_hash'])) {
            Session::flash('error', 'Invalid email or password.');
            header('Location: login.php');
            exit;
        }

        // Set user session
        Session::setUser($user);

        // Get user's organizations
        $organizations = $this->user->getOrganizations($user['id']);

        if (!empty($organizations)) {
            // Set first organization as active
            Session::setOrganization($organizations[0]);
        } else {
            // No organizations found - user needs to create one
            // Don't redirect to settings here to avoid loop, let index.php handle it
        }

        // Remember me
        if ($remember) {
            $token = bin2hex(random_bytes(32));
            $this->user->updateRememberToken($user['id'], $token);
            setcookie('remember_token', $token, time() + (86400 * 30), '/'); // 30 days
        }

        Session::flash('success', 'Welcome back, ' . $user['name'] . '!');
        
        // Redirect to settings if no organization, otherwise to dashboard
        if (empty($organizations)) {
            header('Location: settings.php');
        } else {
            header('Location: index.php');
        }
        exit;
    }

    // Show register form
    public function showRegister()
    {
        if (Session::isAuthenticated()) {
            header('Location: index.php');
            exit;
        }

        $title = "Register";
        include __DIR__ . '/../../views/layouts/auth.php';
        include __DIR__ . '/../../views/auth/register.php';
        include __DIR__ . '/../../views/layouts/auth-footer.php';
    }

    // Handle registration
    public function register()
    {
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $password_confirm = $_POST['password_confirm'] ?? '';
        $organization_name = trim($_POST['organization_name'] ?? '');

        // Validation
        $errors = [];

        if (empty($name)) {
            $errors[] = 'Name is required.';
        }

        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Valid email is required.';
        } else {
            $existing = $this->user->findByEmail($email);
            if ($existing) {
                $errors[] = 'Email already registered.';
            }
        }

        if (empty($password) || strlen($password) < 8) {
            $errors[] = 'Password must be at least 8 characters.';
        }

        if ($password !== $password_confirm) {
            $errors[] = 'Passwords do not match.';
        }

        if (empty($organization_name)) {
            $errors[] = 'Organization name is required.';
        }

        if (!empty($errors)) {
            Session::flash('errors', $errors);
            header('Location: register.php');
            exit;
        }

        // Create user
        $userId = $this->user->create([
            'name' => $name,
            'email' => $email,
            'password' => $password
        ]);

        // Create organization
        $slug = Organization::generateSlug($organization_name);
        // Ensure unique slug
        $originalSlug = $slug;
        $counter = 1;
        while ($this->organization->findBySlug($slug)) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        $organizationId = $this->organization->create([
            'name' => $organization_name,
            'slug' => $slug
        ]);

        // Add user as owner
        $this->organization->addUser($organizationId, $userId, 'owner');

        // Set sessions
        $user = $this->user->findById($userId);
        Session::setUser($user);

        $organization = $this->organization->findById($organizationId);
        Session::setOrganization($organization);

        Session::flash('success', 'Account created successfully! Welcome to Payment System.');
        header('Location: index.php');
        exit;
    }

    // Handle logout
    public function logout()
    {
        Session::destroy();
        setcookie('remember_token', '', time() - 3600, '/');
        Session::flash('success', 'You have been logged out.');
        header('Location: login.php');
        exit;
    }
}

