<?php

class PasswordResetController
{
    private $pdo;
    private $user;
    private $passwordReset;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->user = new User($pdo);
        $this->passwordReset = new PasswordReset($pdo);
        Session::start();
    }

    // Show forgot password form
    public function showForgotPassword()
    {
        $title = "Forgot Password";
        include __DIR__ . '/../../views/layouts/auth.php';
        include __DIR__ . '/../../views/auth/forgot-password.php';
        include __DIR__ . '/../../views/layouts/auth-footer.php';
    }

    // Handle forgot password request
    public function requestReset()
    {
        $email = trim($_POST['email'] ?? '');

        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            Session::flash('error', 'Please enter a valid email address.');
            header('Location: forgot-password.php');
            exit;
        }

        $user = $this->user->findByEmail($email);

        // Always show success message (security best practice)
        Session::flash('success', 'If an account exists with that email, a password reset link has been sent.');

        if ($user) {
            $token = PasswordReset::generateToken();
            $this->passwordReset->create($email, $token);

            // In production, send email here
            // For now, we'll show the reset link (remove in production!)
            $resetUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/reset-password.php?token=' . $token;
            
            // TODO: Send email with reset link
            // For development, store in session (remove in production!)
            Session::flash('reset_link', $resetUrl);
        }

        header('Location: forgot-password.php');
        exit;
    }

    // Show reset password form
    public function showResetPassword()
    {
        $token = $_GET['token'] ?? '';

        if (empty($token)) {
            Session::flash('error', 'Invalid reset token.');
            header('Location: login.php');
            exit;
        }

        $reset = $this->passwordReset->findValidToken($token);

        if (!$reset) {
            Session::flash('error', 'Invalid or expired reset token.');
            header('Location: login.php');
            exit;
        }

        $title = "Reset Password";
        include __DIR__ . '/../../views/layouts/auth.php';
        include __DIR__ . '/../../views/auth/reset-password.php';
        include __DIR__ . '/../../views/layouts/auth-footer.php';
    }

    // Handle password reset
    public function resetPassword()
    {
        $token = $_POST['token'] ?? '';
        $password = $_POST['password'] ?? '';
        $password_confirm = $_POST['password_confirm'] ?? '';

        if (empty($token)) {
            Session::flash('error', 'Invalid reset token.');
            header('Location: login.php');
            exit;
        }

        $reset = $this->passwordReset->findValidToken($token);

        if (!$reset) {
            Session::flash('error', 'Invalid or expired reset token.');
            header('Location: login.php');
            exit;
        }

        if (empty($password) || strlen($password) < 8) {
            Session::flash('error', 'Password must be at least 8 characters.');
            header('Location: reset-password.php?token=' . $token);
            exit;
        }

        if ($password !== $password_confirm) {
            Session::flash('error', 'Passwords do not match.');
            header('Location: reset-password.php?token=' . $token);
            exit;
        }

        // Update password
        $user = $this->user->findByEmail($reset['email']);
        if ($user) {
            $this->user->update($user['id'], ['password' => $password]);
        }

        // Mark token as used
        $this->passwordReset->markAsUsed($token);

        Session::flash('success', 'Password reset successfully. Please login with your new password.');
        header('Location: login.php');
        exit;
    }
}

