<?php

class SettingsController
{
    private $pdo;
    private $organization;
    private $telegramAccess;
    private $subscription;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->organization = new Organization($pdo);
        $this->telegramAccess = new TelegramAccess($pdo);
        $this->subscription = new Subscription($pdo);
    }

    // Show settings page
    public function index()
    {
        $organization = Session::getOrganization();
        
        // Try to load organization from database if not in session
        if (!$organization) {
            $user = Session::getUser();
            if ($user) {
                $userModel = new User($this->pdo);
                $organizations = $userModel->getOrganizations($user['id']);
                
                if (!empty($organizations)) {
                    Session::setOrganization($organizations[0]);
                    $organization = Session::getOrganization();
                }
            }
        }
        
        // Get organization data if exists
        $orgData = [];
        $telegramAccessList = [];
        $currentSubscription = null;
        
        if ($organization) {
            $orgData = $this->organization->findById($organization['id']);
            $telegramAccessList = $this->telegramAccess->getByOrganization($organization['id']);
            $currentSubscription = $this->subscription->getActive($organization['id']);
        } else {
            // If no organization, create empty orgData for form
            $orgData = [
                'name' => '',
                'telegram_bot_token' => ''
            ];
        }

        $title = "Settings";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/settings/index.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Update organization settings
    public function updateOrganization()
    {
        $organization = Session::getOrganization();
        $user = Session::getUser();
        
        if (!$user) {
            Session::flash('error', 'User not found.');
            header('Location: login.php');
            exit;
        }

        $name = trim($_POST['name'] ?? '');
        $telegramBotToken = trim($_POST['telegram_bot_token'] ?? '');

        if (empty($name)) {
            Session::flash('error', 'Organization name is required.');
            header('Location: settings.php');
            exit;
        }

        // If no organization exists, create one
        if (!$organization) {
            $slug = Organization::generateSlug($name);
            // Ensure unique slug
            $originalSlug = $slug;
            $counter = 1;
            while ($this->organization->findBySlug($slug)) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }

            $organizationId = $this->organization->create([
                'name' => $name,
                'slug' => $slug,
                'telegram_bot_token' => !empty($telegramBotToken) ? $telegramBotToken : null
            ]);

            // Add user as owner
            $this->organization->addUser($organizationId, $user['id'], 'owner');

            // Set in session
            $newOrg = $this->organization->findById($organizationId);
            Session::setOrganization($newOrg);
            
            Session::flash('success', 'Organization created successfully!');
            header('Location: index.php');
            exit;
        } else {
            // Update existing organization
            $data = ['name' => $name];
            if (!empty($telegramBotToken)) {
                $data['telegram_bot_token'] = $telegramBotToken;
            }

            if ($this->organization->update($organization['id'], $data)) {
                // Update session
                $updated = $this->organization->findById($organization['id']);
                Session::setOrganization($updated);
                
                Session::flash('success', 'Organization settings updated successfully.');
            } else {
                Session::flash('error', 'Failed to update settings.');
            }

            header('Location: settings.php');
            exit;
        }
    }

    // Add Telegram access
    public function addTelegramAccess()
    {
        $organization = Session::getOrganization();
        if (!$organization) {
            Session::flash('error', 'Please create an organization first.');
            header('Location: settings.php');
            exit;
        }

        $chatId = trim($_POST['chat_id'] ?? '');
        $chatType = $_POST['chat_type'] ?? 'user';
        $chatTitle = trim($_POST['chat_title'] ?? '');

        if (empty($chatId)) {
            Session::flash('error', 'Chat ID is required.');
            header('Location: settings.php');
            exit;
        }

        if ($this->telegramAccess->create([
            'organization_id' => $organization['id'],
            'chat_id' => $chatId,
            'chat_type' => $chatType,
            'chat_title' => $chatTitle
        ])) {
            Session::flash('success', 'Telegram access added successfully.');
        } else {
            Session::flash('error', 'Failed to add Telegram access. Chat ID may already exist.');
        }

        header('Location: settings.php');
        exit;
    }

    // Delete Telegram access
    public function deleteTelegramAccess()
    {
        $organization = Session::getOrganization();
        if (!$organization) {
            Session::flash('error', 'No organization found.');
            header('Location: settings.php');
            exit;
        }

        $id = $_POST['id'] ?? null;
        if (!$id) {
            Session::flash('error', 'Invalid access ID.');
            header('Location: settings.php');
            exit;
        }

        if ($this->telegramAccess->delete($id)) {
            Session::flash('success', 'Telegram access removed successfully.');
        } else {
            Session::flash('error', 'Failed to remove Telegram access.');
        }

        header('Location: settings.php');
        exit;
    }

    // Toggle Telegram access
    public function toggleTelegramAccess()
    {
        $organization = Session::getOrganization();
        if (!$organization) {
            Session::flash('error', 'No organization found.');
            header('Location: settings.php');
            exit;
        }

        $id = $_POST['id'] ?? null;
        if (!$id) {
            Session::flash('error', 'Invalid access ID.');
            header('Location: settings.php');
            exit;
        }

        if ($this->telegramAccess->toggleActive($id)) {
            Session::flash('success', 'Telegram access updated successfully.');
        } else {
            Session::flash('error', 'Failed to update Telegram access.');
        }

        header('Location: settings.php');
        exit;
    }
}

