<?php

class User
{
    private $pdo;
    private $table = "users";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Create new user
    public function create($data)
    {
        $sql = "INSERT INTO {$this->table} (name, email, password_hash) 
                VALUES (:name, :email, :password_hash)";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':name' => $data['name'],
            ':email' => $data['email'],
            ':password_hash' => password_hash($data['password'], PASSWORD_DEFAULT)
        ]);
        
        return $this->pdo->lastInsertId();
    }

    // Find by email
    public function findByEmail($email)
    {
        $sql = "SELECT * FROM {$this->table} WHERE email = :email LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':email' => $email]);
        return $stmt->fetch();
    }

    // Find by ID
    public function findById($id)
    {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch();
    }

    // Update user
    public function update($id, $data)
    {
        $fields = [];
        $params = [':id' => $id];

        if (isset($data['name'])) {
            $fields[] = "name = :name";
            $params[':name'] = $data['name'];
        }
        if (isset($data['email'])) {
            $fields[] = "email = :email";
            $params[':email'] = $data['email'];
        }
        if (isset($data['password'])) {
            $fields[] = "password_hash = :password_hash";
            $params[':password_hash'] = password_hash($data['password'], PASSWORD_DEFAULT);
        }

        if (empty($fields)) {
            return false;
        }

        $sql = "UPDATE {$this->table} SET " . implode(", ", $fields) . " WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    // Delete user
    public function delete($id)
    {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

    // Get all users
    public function all($limit = null, $offset = 0)
    {
        $sql = "SELECT * FROM {$this->table} ORDER BY created_at DESC";
        if ($limit !== null) {
            $sql .= " LIMIT :limit OFFSET :offset";
        }
        
        $stmt = $this->pdo->prepare($sql);
        if ($limit !== null) {
            $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        }
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // Verify password
    public function verifyPassword($password, $hash)
    {
        return password_verify($password, $hash);
    }

    // Update remember token
    public function updateRememberToken($id, $token)
    {
        $sql = "UPDATE {$this->table} SET remember_token = :token WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':token' => $token, ':id' => $id]);
    }

    // Get user organizations
    public function getOrganizations($userId)
    {
        try {
            $sql = "SELECT o.*, ou.role 
                    FROM organizations o
                    INNER JOIN organization_users ou ON o.id = ou.organization_id
                    WHERE ou.user_id = :user_id
                    ORDER BY ou.created_at DESC";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([':user_id' => $userId]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            // If table doesn't exist yet, return empty array
            if (strpos($e->getMessage(), "doesn't exist") !== false) {
                return [];
            }
            throw $e;
        }
    }
}

