<?php
require_once __DIR__ . '/../routes/web.php';

Authenticate::handle();

$organization = Session::getOrganization();
if (!$organization) {
    // Try to get user's organizations from database
    $user = Session::getUser();
    if ($user) {
        $userModel = new User($pdo);
        $organizations = $userModel->getOrganizations($user['id']);
        
        if (!empty($organizations)) {
            // Set first organization as active
            Session::setOrganization($organizations[0]);
            $organization = Session::getOrganization();
        }
    }
    
    // If still no organization, redirect to settings (but don't create infinite loop)
    if (!$organization && basename($_SERVER['PHP_SELF']) !== 'settings.php') {
        Session::flash('error', 'No organization found. Please configure your organization settings.');
        header('Location: settings.php');
        exit;
    }
}

$summary = new SummaryController($pdo);
$dashboardStats = $summary->getDashboardStats($organization['id']);

$title = "Dashboard";
include __DIR__ . '/../views/layouts/main.php';
?>

<div class="mb-8">
    <h1 class="text-3xl font-bold text-mono-900 mb-2">Dashboard Overview</h1>
    <p class="text-mono-600">Real-time payment statistics and insights</p>
</div>

<!-- Statistics Cards -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
    <!-- Today -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-sm font-medium text-mono-600">Today</h3>
            <span class="text-2xl">📅</span>
        </div>
        <div class="space-y-1">
            <p class="text-2xl font-bold text-mono-900">$<?= number_format($dashboardStats['today']['total_amount'] ?? 0, 2) ?></p>
            <p class="text-sm text-mono-500"><?= $dashboardStats['today']['total_count'] ?? 0 ?> transactions</p>
        </div>
    </div>

    <!-- This Week -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-sm font-medium text-mono-600">This Week</h3>
            <span class="text-2xl">📆</span>
        </div>
        <div class="space-y-1">
            <p class="text-2xl font-bold text-mono-900">$<?= number_format($dashboardStats['thisWeek']['total_amount'] ?? 0, 2) ?></p>
            <p class="text-sm text-mono-500"><?= $dashboardStats['thisWeek']['total_count'] ?? 0 ?> transactions</p>
        </div>
    </div>

    <!-- This Month -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-sm font-medium text-mono-600">This Month</h3>
            <span class="text-2xl">🗓️</span>
        </div>
        <div class="space-y-1">
            <p class="text-2xl font-bold text-mono-900">$<?= number_format($dashboardStats['thisMonth']['total_amount'] ?? 0, 2) ?></p>
            <p class="text-sm text-mono-500"><?= $dashboardStats['thisMonth']['total_count'] ?? 0 ?> transactions</p>
        </div>
    </div>

    <!-- This Year -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-sm font-medium text-mono-600">This Year</h3>
            <span class="text-2xl">📘</span>
        </div>
        <div class="space-y-1">
            <p class="text-2xl font-bold text-mono-900">$<?= number_format($dashboardStats['thisYear']['total_amount'] ?? 0, 2) ?></p>
            <p class="text-sm text-mono-500"><?= $dashboardStats['thisYear']['total_count'] ?? 0 ?> transactions</p>
        </div>
    </div>
</div>

<!-- Additional Stats -->
<div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
    <!-- All Time Stats -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <h3 class="text-lg font-semibold text-mono-900 mb-4">All Time Statistics</h3>
        <div class="space-y-3">
            <div class="flex justify-between items-center">
                <span class="text-mono-600">Total Amount</span>
                <span class="font-semibold text-mono-900">$<?= number_format($dashboardStats['allTime']['total_amount'] ?? 0, 2) ?></span>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-mono-600">Total Transactions</span>
                <span class="font-semibold text-mono-900"><?= number_format($dashboardStats['allTime']['total_count'] ?? 0) ?></span>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-mono-600">Average Amount</span>
                <span class="font-semibold text-mono-900">$<?= number_format($dashboardStats['allTime']['avg_amount'] ?? 0, 2) ?></span>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-mono-600">Highest Transaction</span>
                <span class="font-semibold text-mono-900">$<?= number_format($dashboardStats['allTime']['max_amount'] ?? 0, 2) ?></span>
            </div>
        </div>
    </div>

    <!-- Bank Distribution -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <h3 class="text-lg font-semibold text-mono-900 mb-4">By Bank</h3>
        <?php if (!empty($dashboardStats['byBank'])): ?>
            <div class="space-y-3">
                <?php foreach ($dashboardStats['byBank'] as $bank): ?>
                    <div>
                        <div class="flex justify-between items-center mb-1">
                            <span class="text-mono-700 font-medium"><?= htmlspecialchars($bank['bank']) ?></span>
                            <span class="text-mono-900 font-semibold">$<?= number_format($bank['total'], 2) ?></span>
                        </div>
                        <div class="w-full bg-mono-200 rounded-full h-2">
                            <div class="bg-mono-600 h-2 rounded-full" style="width: <?= $dashboardStats['allTime']['total_amount'] > 0 ? ($bank['total'] / $dashboardStats['allTime']['total_amount'] * 100) : 0 ?>%"></div>
                        </div>
                        <p class="text-xs text-mono-500 mt-1"><?= $bank['count'] ?> transactions</p>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <p class="text-mono-500 text-sm">No bank data available</p>
        <?php endif; ?>
    </div>

    <!-- Top Merchants -->
    <div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
        <h3 class="text-lg font-semibold text-mono-900 mb-4">Top Merchants</h3>
        <?php if (!empty($dashboardStats['topMerchants'])): ?>
            <div class="space-y-3">
                <?php foreach ($dashboardStats['topMerchants'] as $merchant): ?>
                    <div class="flex justify-between items-center">
                        <div class="flex-1">
                            <p class="text-mono-700 font-medium"><?= htmlspecialchars($merchant['merchant'] ?: 'Unknown') ?></p>
                            <p class="text-xs text-mono-500"><?= $merchant['count'] ?> transactions</p>
                        </div>
                        <span class="text-mono-900 font-semibold ml-4">$<?= number_format($merchant['total'], 2) ?></span>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <p class="text-mono-500 text-sm">No merchant data available</p>
        <?php endif; ?>
    </div>
</div>

<!-- Recent Transactions -->
<div class="bg-white rounded-lg shadow-sm border border-mono-200 p-6">
    <div class="flex justify-between items-center mb-4">
        <h3 class="text-lg font-semibold text-mono-900">Recent Transactions</h3>
        <a href="transaction-list.php" class="text-sm text-mono-600 hover:text-mono-900">View All →</a>
    </div>
    <?php if (!empty($dashboardStats['recent'])): ?>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b border-mono-200">
                        <th class="text-left py-3 px-4 text-sm font-semibold text-mono-700">Date & Time</th>
                        <th class="text-left py-3 px-4 text-sm font-semibold text-mono-700">Payer</th>
                        <th class="text-left py-3 px-4 text-sm font-semibold text-mono-700">Bank</th>
                        <th class="text-left py-3 px-4 text-sm font-semibold text-mono-700">Merchant</th>
                        <th class="text-right py-3 px-4 text-sm font-semibold text-mono-700">Amount</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($dashboardStats['recent'] as $tx): ?>
                        <tr class="border-b border-mono-100 hover:bg-mono-50">
                            <td class="py-3 px-4 text-sm text-mono-600">
                                <?= date("M j, Y", strtotime($tx['date'])) ?><br>
                                <span class="text-xs text-mono-500"><?= date("g:i A", strtotime($tx['time'])) ?></span>
                            </td>
                            <td class="py-3 px-4 text-sm text-mono-700"><?= htmlspecialchars($tx['payer_name']) ?></td>
                            <td class="py-3 px-4">
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-mono-200 text-mono-800">
                                    <?= htmlspecialchars($tx['bank']) ?>
                                </span>
                            </td>
                            <td class="py-3 px-4 text-sm text-mono-600"><?= htmlspecialchars($tx['merchant'] ?: 'N/A') ?></td>
                            <td class="py-3 px-4 text-right text-sm font-semibold text-mono-900">$<?= number_format($tx['amount'], 2) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <p class="text-mono-500 text-sm py-8 text-center">No recent transactions</p>
    <?php endif; ?>
</div>

<?php include __DIR__ . '/../views/layouts/footer.php'; ?>
